<?php /** @noinspection PhpUndefinedClassInspection */
/**
 * Ader for Elementor
 * Customizable add to cart button for Elementor editor
 * Exclusively on https://1.envato.market/ader-elementor
 *
 * @encoding        UTF-8
 * @version         1.0.1
 * @copyright       (C) 2018 - 2021 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Dmitry Merkulov (dmitry@merkulov.design), Cherviakov Vlad (vladchervjakov@gmail.com)
 * @support         help@merkulov.design
 **/

namespace Merkulove\AderElementor;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

use Exception;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Core\Schemes\Typography;
use Elementor\Core\Schemes\Color;
use Merkulove\AderElementor\Unity\Plugin as UnityPlugin;

/** @noinspection PhpUnused */
/**
 * Product Add to cart - Custom Elementor Widget.
 **/
class product_add_to_cart_elementor extends Widget_Base {

    /**
     * Use this to sort widgets.
     * A smaller value means earlier initialization of the widget.
     * Can take negative values.
     * Default widgets and widgets from 3rd party developers have 0 $mdp_order
     **/
    public $mdp_order = 1;

    /**
     * Widget base constructor.
     * Initializing the widget base class.
     *
     * @access public
     * @throws Exception If arguments are missing when initializing a full widget instance.
     * @param array      $data Widget data. Default is an empty array.
     * @param array|null $args Optional. Widget default arguments. Default is null.
     *
     * @return void
     **/
    public function __construct( $data = [], $args = null ) {

        parent::__construct( $data, $args );

        wp_register_style( 'mdp-ader-elementor-admin', UnityPlugin::get_url() . 'css/elementor-admin' . UnityPlugin::get_suffix() . '.css', [], UnityPlugin::get_version() );
        wp_register_style( 'mdp-ader-elementor', UnityPlugin::get_url() . 'css/ader-elementor' . UnityPlugin::get_suffix() . '.css', [], UnityPlugin::get_version() );

        // include required frontend files for woocommerce in editor
        if ( ! empty( $_REQUEST['action'] ) && 'elementor' === $_REQUEST['action'] && is_admin() && $this->is_active_woocommerce() ) {
            wc()->frontend_includes();
        }

    }

    /**
     * Return a widget name.
     *
     * @return string
     **/
    public function get_name() {

        return 'mdp-ader-add-to-cart-elementor';

    }

    /**
     * Return the widget title that will be displayed as the widget label.
     *
     * @return string
     **/
    public function get_title() {

        return esc_html__( 'Add To Cart', 'ader-elementor' );

    }

    /**
     * Set the widget icon.
     *
     * @return string
     */
    public function get_icon() {

        return 'mdp-product-add-to-cart-elementor-widget-icon';

    }

    /**
     * Set the category of the widget.
     *
     * @return array with category names
     **/
    public function get_categories() {

        return [ 'general' ];

    }

    /**
     * Get widget keywords. Retrieve the list of keywords the widget belongs to.
     *
     * @access public
     *
     * @return array Widget keywords.
     **/
    public function get_keywords() {

        return [ 'Merkulove', 'Ader', 'cart', 'add', 'woocommerce' ];

    }

    /**
     * Get style dependencies.
     * Retrieve the list of style dependencies the widget requires.
     *
     * @access public
     *
     * @return array Widget styles dependencies.
     **/
    public function get_style_depends() {

        return [ 'mdp-ader-elementor', 'mdp-ader-elementor-admin', 'elementor-icons-fa-solid' ];

    }

	/**
	 * Get script dependencies.
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @access public
     *
	 * @return array Element scripts dependencies.
	 **/
	public function get_script_depends() {

		return [ 'mdp-ader' ];

    }

    /**
     * Add the widget controls.
     *
     * @access protected
     * @return void with category names
     **/
    protected function _register_controls() {

        /** Content Tab. */
        $this->tab_content();

        /** Style Tab. */
        $this->tab_style();

    }

    /**
     * Add widget controls on Content tab.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function tab_content() {

    }

    /**
     * Add widget controls on Style tab.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function tab_style() {

        /** Style -> Section Style Button. */
        $this->section_style_button();

        /** Style -> Section Style Quantity. */
        $this->section_style_quantity();

        /** Style -> Section Style Variations. */
        $this->section_style_variations();

    }

    /**
     * Checks if woocommerce is active
     *
     * @since 1.0.0
     * @access private
     *
     * @return bool
     **/
    private function is_active_woocommerce() {

        $woocommerce_active = false;
        if (  in_array('woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
            $woocommerce_active = true;
        }

        return $woocommerce_active;
    }

    /**
     * Function for generating margin padding controls.
     *
     * @param $section_id
     * @param $html_class
     * @return void
     * @since 1.0.0
     * @access private
     *
     */
    private function generate_margin_padding_controls( $section_id, $html_class ) {
        $this->add_responsive_control(
            $section_id.'_margin',
            [
                'label' => esc_html__( 'Margin', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'selectors' => [
                    "{{WRAPPER}} .$html_class" => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important;',
                ],
            ]
        );

        $this->add_responsive_control(
            $section_id.'_padding',
            [
                'label' => esc_html__( 'Padding', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'separator' => 'after',
                'selectors' => [
                    "{{WRAPPER}} .$html_class" => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important;',
                ],
            ]
        );
    }

    /**
     * Function for generating typography and tabs controls.
     *
     * @param $section_id
     * @param $opts
     * @return void
     * @since 1.0.0
     * @access private
     */
    private function generate_typography_tabs_controls($section_id, $opts = [] ) {
        $style_opts = [
            'html_class' => array_key_exists( 'html_class', $opts ) ? $opts['html_class'] : '',
            'focus_class' => array_key_exists( 'focus_class', $opts ) ? $opts['focus_class'] : '',
            'include_color' => array_key_exists( 'include_color', $opts ) ? $opts['include_color'] : true,
            'include_bg' => array_key_exists( 'include_bg', $opts ) ? $opts['include_color'] : true,
            'include_typography' => array_key_exists( 'include_typography', $opts ) ? $opts['include_typography'] : true,
            'include_transition' => array_key_exists( 'include_transition', $opts ) ? $opts['include_transition'] : true,
            'additional_color' => array_key_exists( 'additional_color', $opts ) ? $opts['additional_color'] : false,
            'include_focus_tab' => array_key_exists( 'include_focus_tab', $opts ) ? $opts['include_focus_tab'] : false,
            'color_prefix' => array_key_exists( 'color_prefix', $opts ) ? $opts['color_prefix'] : '',
            'color_class' => array_key_exists( 'color_class', $opts ) ? $opts['color_class'] : '',
            'color_hover_class' => array_key_exists( 'color_hover_class', $opts ) ? $opts['color_hover_class'] : '',
            'color_focus_class' => array_key_exists( 'color_focus_class', $opts ) ? $opts['color_focus_class'] : '',
            'color_hover_selector' => array_key_exists( 'color_hover_selector', $opts ) ? $opts['color_hover_selector'] : '',
            'additional_color_name' => array_key_exists( 'additional_color_name', $opts ) ? $opts['additional_color_name'] : '',
            'additional_color_class' => array_key_exists( 'additional_color_class', $opts ) ? $opts['additional_color_class'] : '',
            'additional_color_hover_class' => array_key_exists( 'additional_color_hover_class', $opts ) ? $opts['additional_color_hover_class'] : '',
            'additional_color_focus_class' => array_key_exists( 'additional_color_focus_class', $opts ) ? $opts['additional_color_focus_class'] : '',
            'additional_transition_selector' => array_key_exists( 'additional_transition_selector', $opts ) ? $opts['additional_transition_selector'] : '',
            'typography_class' => array_key_exists( 'typography_class', $opts ) ? $opts['typography_class'] : '',
            'color_scheme_default' => array_key_exists( 'color_scheme_default', $opts ) ? $opts['color_scheme_default'] : Color::COLOR_3,
            'additional_color_scheme_default' => array_key_exists( 'additional_color_scheme_default', $opts ) ? $opts['additional_color_scheme_default'] : Color::COLOR_3
        ];


        if ( $style_opts['include_typography'] ) {
            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => $section_id . '_typography',
                    'label' => esc_html__('Typography', 'ader-elementor'),
                    'scheme' => Typography::TYPOGRAPHY_1,
                    'selector' => "{{WRAPPER}} .".$style_opts['typography_class'],
                ]
            );
        }

        $this->start_controls_tabs( $section_id.'_style_tabs' );

        $this->start_controls_tab( $section_id.'_normal_style_tab', ['label' => esc_html__( 'NORMAL', 'ader-elementor' )] );

        if ( $style_opts['include_color'] ) {

            $this->add_control(
                $section_id . '_normal_text_color',
                [
                    'label' => esc_html__($style_opts['color_prefix'].'Color', 'ader-elementor'),
                    'type' => Controls_Manager::COLOR,
                    'scheme' => [
                        'type' => Color::get_type(),
                        'value' => $style_opts['color_scheme_default'],
                    ],
                    'selectors' => [
                        "{{WRAPPER}} .".$style_opts['color_class'] => 'color: {{VALUE}} !important;',
                    ],
                ]
            );

        }

        if ( $style_opts['additional_color'] ) {
            $this->add_control(
                $section_id . '_' . $style_opts['additional_color_name'] . '_normal_text_color',
                [
                    'label' => esc_html__( $style_opts['additional_color_name'], 'ader-elementor' ),
                    'type' => Controls_Manager::COLOR,
                    'scheme' => [
                        'type' => Color::get_type(),
                        'value' => $style_opts['additional_color_scheme_default'],
                    ],
                    'selectors' => [
                        "{{WRAPPER}} .".$style_opts['additional_color_class'] => 'color: {{VALUE}} !important;',

                    ],
                ]
            );
        }

        if ( $style_opts['include_bg'] ) {

            $this->add_group_control(
                Group_Control_Background::get_type(),
                [
                    'name' => $section_id . '_normal_background',
                    'label' => esc_html__('Background type', 'ader-elementor'),
                    'types' => ['classic', 'gradient', 'video'],
                    'selector' => "{{WRAPPER}} .".$style_opts['html_class'],
                ]
            );

        }

        $this->add_control(
            $section_id . '_separate_normal',
            [
                'type' => Controls_Manager::DIVIDER,
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => $section_id.'_border_normal',
                'label' => esc_html__( 'Border Type', 'ader-elementor' ),
                'selector' => "{{WRAPPER}} .".$style_opts['html_class'],
            ]
        );

        $this->add_responsive_control(
            $section_id.'_border_radius_normal',
            [
                'label' => esc_html__( 'Border radius', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    "{{WRAPPER}} .".$style_opts['html_class'] => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => $section_id.'_box_shadow_normal',
                'label' => esc_html__( 'Box Shadow', 'ader-elementor' ),
                'selector' => "{{WRAPPER}} .".$style_opts['html_class'],
            ]
        );


        $this->end_controls_tab();

        $this->start_controls_tab( $section_id.'_hover_style_tab', ['label' => esc_html__( 'HOVER', 'ader-elementor' )] );

        if ( $style_opts['include_color'] ) {
            $this->add_control(
                $section_id . '_hover_color',
                [
                    'label' => esc_html__($style_opts['color_prefix'].'Color', 'ader-elementor'),
                    'type' => Controls_Manager::COLOR,
                    'scheme' => [
                        'type' => Color::get_type(),
                        'value' => $style_opts['color_scheme_default'],
                    ],
                    'selectors' => [
                        "{{WRAPPER}} .".$style_opts['color_hover_class'] => 'color: {{VALUE}} !important;',
                    ],
                ]
            );
        }

        if ( $style_opts['additional_color'] ) {
            $this->add_control(
                $section_id . '_' . $style_opts['additional_color_name'] . '_hover_text_color',
                [
                    'label' => esc_html__( $style_opts['additional_color_name'], 'ader-elementor'),
                    'type' => Controls_Manager::COLOR,
                    'scheme' => [
                        'type' => Color::get_type(),
                        'value' => $style_opts['additional_color_scheme_default'],
                    ],
                    'selectors' => [
                        "{{WRAPPER}} .".$style_opts['additional_color_hover_class'] => 'color: {{VALUE}} !important;',
                    ],
                ]
            );
        }

        if ( $style_opts['include_bg'] ) {
            $this->add_group_control(
                Group_Control_Background::get_type(),
                [
                    'name' => $section_id . '_background_hover',
                    'label' => esc_html__('Background type', 'ader-elementor'),
                    'types' => ['classic', 'gradient', 'video'],
                    'selector' => "{{WRAPPER}} .".$style_opts['html_class'].":hover",
                ]
            );
        }

        if ( $style_opts['include_transition'] ) {
            $this->add_control(
                $section_id.'_hover_transition',
                [
                    'label' => esc_html__( 'Hover transition duration', 'ader-elementor' ),
                    'type' => Controls_Manager::SLIDER,
                    'size_units' => [ 's' ],
                    'range' => [
                        's' => [
                            'min' => 0.1,
                            'max' => 5,
                            'step' => 0.1,
                        ],
                    ],
                    'default' => [
                        'unit' => 's',
                        'size' => 0,
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .'.$style_opts['html_class'] => 'transition: color {{SIZE}}{{UNIT}}, background {{SIZE}}{{UNIT}}, box-shadow {{SIZE}}{{UNIT}}, border-radius {{SIZE}}{{UNIT}}, border {{SIZE}}{{UNIT}}, filter {{SIZE}}{{UNIT}}, stroke {{SIZE}}{{UNIT}};',
                        $style_opts['additional_transition_selector'] => 'transition: color {{SIZE}}{{UNIT}}, background {{SIZE}}{{UNIT}}, box-shadow {{SIZE}}{{UNIT}}, border-radius {{SIZE}}{{UNIT}}, border {{SIZE}}{{UNIT}}, filter {{SIZE}}{{UNIT}}, stroke {{SIZE}}{{UNIT}};;'
                    ],
                ]
            );
        }

        $this->add_control(
            $section_id.'_separate_hover',
            [
                'type' => Controls_Manager::DIVIDER,
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => $section_id.'_border_hover',
                'label' => esc_html__( 'Border Type', 'ader-elementor' ),
                'selector' => "{{WRAPPER}} .".$style_opts['html_class'].":hover",
            ]
        );

        $this->add_responsive_control(
            $section_id.'_border_radius_hover',
            [
                'label' => esc_html__( 'Border radius', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    "{{WRAPPER}} .".$style_opts['html_class'].":hover" => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => $section_id.'_box_shadow_hover',
                'label' => esc_html__( 'Box Shadow', 'ader-elementor' ),
                'selector' => "{{WRAPPER}} .".$style_opts['html_class'].":hover",
            ]
        );

        $this->end_controls_tab();

        if ( $style_opts['include_focus_tab'] ) {

            $this->start_controls_tab( $section_id . '_focus_style_tab', ['label' => esc_html__( 'FOCUS', 'ader-elementor' )] );

            if ( $style_opts['include_color'] ) {
                $this->add_control(
                    $section_id . '_active_color',
                    [
                        'label' => esc_html__( $style_opts['color_prefix'] . 'Color', 'ader-elementor '),
                        'type' => Controls_Manager::COLOR,
                        'scheme' => [
                            'type' => Color::get_type(),
                            'value' => $style_opts['color_scheme_default'],
                        ],
                        'selectors' => [
                            "{{WRAPPER}} ." . $style_opts['color_focus_class'] => 'color: {{VALUE}} !important;',
                        ],
                    ]
                );
            }

            if ( $style_opts['additional_color'] ) {
                $this->add_control(
                    $section_id . '_' . $style_opts['additional_color_name'] . '_active_text_color',
                    [
                        'label' => esc_html__( $style_opts['additional_color_name'], 'ader-elementor' ),
                        'type' => Controls_Manager::COLOR,
                        'scheme' => [
                            'type' => Color::get_type(),
                            'value' => $style_opts['additional_color_scheme_default'],
                        ],
                        'selectors' => [
                            "{{WRAPPER}} ." . $style_opts['additional_color_focus_class'] => 'color: {{VALUE}} !important;',
                        ],
                    ]
                );
            }

            if ( $style_opts['include_bg'] ) {
                $this->add_group_control(
                    Group_Control_Background::get_type(),
                    [
                        'name' => $section_id . '_background_active',
                        'label' => esc_html__( 'Background type', 'ader-elementor' ),
                        'types' => ['classic', 'gradient', 'video'],
                        'selector' => "{{WRAPPER}} ." . $style_opts['focus_class'],
                    ]
                );
            }

            $this->add_control(
                $section_id . '_separate_active',
                [
                    'type' => Controls_Manager::DIVIDER,
                ]
            );

            $this->add_group_control(
                Group_Control_Border::get_type(),
                [
                    'name' => $section_id . '_border_active',
                    'label' => esc_html__( 'Border Type', 'ader-elementor' ),
                    'selector' => "{{WRAPPER}} ." . $style_opts['focus_class'],
                ]
            );

            $this->add_responsive_control(
                $section_id . '_border_radius_active',
                [
                    'label' => esc_html__( 'Border radius', 'ader-elementor' ),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => ['px', '%', 'em'],
                    'selectors' => [
                        "{{WRAPPER}} ." . $style_opts['focus_class'] => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                ]
            );

            $this->add_group_control(
                Group_Control_Box_Shadow::get_type(),
                [
                    'name' => $section_id . '_box_shadow_active',
                    'label' => esc_html__( 'Box Shadow', 'ader-elementor' ),
                    'selector' => "{{WRAPPER}} ." . $style_opts['focus_class'],
                ]
            );

            $this->end_controls_tab();

        }


        $this->end_controls_tabs();
    }

    /**
     * Add widget controls: Style -> Section Style Button.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_button() {

        $this->start_controls_section( 'section_style_button', [
            'label' => esc_html__( 'Button', 'ader-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE
        ] );

        $this->generate_margin_padding_controls( 'section_style_button', 'cart .button' );

        $this->add_control(
            'button_alignment',
            [
                'label' => esc_html__( 'Button alignment', 'ader-elementor' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'ader-elementor' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'ader-elementor' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'ader-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                    'justify' => [
                        'title' => esc_html__( 'Justify', 'ader-elementor' ),
                        'icon' => 'fa fa-align-justify',
                    ],
                ],
                'default' => 'center',
                'toggle' => true,
            ]
        );

        $style_opts = [
            'html_class' => 'cart .button',
            'color_class' => 'cart .button',
            'typography_class' => 'cart .button',
            'color_hover_class' => 'cart .button:hover'
        ];

        $this->generate_typography_tabs_controls( 'section_style_button', $style_opts );

        $this->end_controls_section();

    }

    /**
     * Add widget controls: Style -> Section Style Quantity.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_quantity() {

        $this->start_controls_section('section_style_quantity', [
            'label' => esc_html__('Quantity', 'ader-elementor'),
            'tab' => Controls_Manager::TAB_STYLE
        ]);

        $this->generate_margin_padding_controls( 'section_style_quantity', 'quantity .qty' );

        $this->add_control(
            'quantity_spacing_separate_normal',
            [
                'type' => Controls_Manager::DIVIDER,
            ]
        );

        $style_opts = [
            'html_class' => 'quantity .qty',
            'color_class' => 'quantity .qty',
            'typography_class' => 'quantity .qty',
            'color_hover_class' => 'quantity .qty:hover',
            'additional_transition_selector' => 'quantity .qty',
            'focus_class' => 'quantity .qty:focus',
            'color_focus_class' => 'quantity .qty:focus',
            'include_focus_tab' => true,
        ];

        $this->generate_typography_tabs_controls( 'section_style_quantity', $style_opts );


        $this->end_controls_section();

    }

    /**
     * Add widget controls: Style -> Section Style Variations.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_variations() {

        $this->start_controls_section('section_style_variations', [
            'label' => esc_html__('Variations', 'ader-elementor'),
            'tab' => Controls_Manager::TAB_STYLE
        ]);

        $this->generate_margin_padding_controls( 'section_style_variations', 'cart .variations' );

        $this->add_control(
            'variations_width',
            [
                'label' => esc_html__( 'Width', 'ader-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 5,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .cart .variations' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'variations_width_separate_normal',
            [
                'type' => Controls_Manager::DIVIDER,
            ]
        );

        $style_opts = [
            'html_class' => 'cart .variations',
            'color_class' => 'cart .variations',
            'typography_class' => 'cart .variations',
            'color_hover_class' => 'cart .variations:hover',
            'additional_color' => true,
            'additional_transition_selector' => '{{WRAPPER}} .variations .value select',
            'additional_color_class' => 'variations .value select',
            'additional_color_hover_class' => 'variations .value select:hover',
            'additional_color_name' => 'Select color',
        ];

        $this->generate_typography_tabs_controls( 'section_style_variations', $style_opts );

        $this->end_controls_section();

    }


        /**
     * Render Frontend Output. Generate the final HTML on the frontend.
     *
     * @access protected
     *
     * @return void
     **/
    protected function render() {
        $settings = $this->get_settings_for_display();


        if ( $this->is_active_woocommerce() ) {

            $product = wc_get_product();

            if ( empty( $product ) ) {
                return;
            }
        }

        ?>
        <?php if ( $this->is_active_woocommerce() ): ?>
        <!-- Start Ader for Elementor WordPress Plugin -->
        <div class="mdp-ader-elementor-box mdp-ader-elementor-product-button-alignment-<?php esc_attr_e( $settings['button_alignment'] ); ?>">
            <?php woocommerce_template_single_add_to_cart(); ?>
        </div>
        <!-- End Ader for Elementor WordPress Plugin -->
        <?php else: ?>
        Install WooCommerce to display the button!
        <?php endif; ?>
	    <?php

    }

    /**
     * Return link for documentation
     * Used to add stuff after widget
     *
     * @access public
     *
     * @return string
     **/
    public function get_custom_help_url() {

        return 'https://docs.merkulov.design/tag/ader';

    }

}
