<?php /** @noinspection PhpUndefinedClassInspection */
/**
 * Ader for Elementor
 * Customizable add to cart button for Elementor editor
 * Exclusively on https://1.envato.market/ader-elementor
 *
 * @encoding        UTF-8
 * @version         1.0.1
 * @copyright       (C) 2018 - 2021 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Dmitry Merkulov (dmitry@merkulov.design), Cherviakov Vlad (vladchervjakov@gmail.com)
 * @support         help@merkulov.design
 **/

namespace Merkulove\AderElementor;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

use Elementor\Icons_Manager;
use Exception;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Core\Schemes\Typography;
use Elementor\Core\Schemes\Color;
use Merkulove\AderElementor\Unity\Plugin as UnityPlugin;

/** @noinspection PhpUnused */
/**
 * Custom add to cart - Custom Elementor Widget.
 **/
class custom_add_to_cart_elementor extends Widget_Base {

    /**
     * Use this to sort widgets.
     * A smaller value means earlier initialization of the widget.
     * Can take negative values.
     * Default widgets and widgets from 3rd party developers have 0 $mdp_order
     **/
    public $mdp_order = 1;

    /**
     * Widget base constructor.
     * Initializing the widget base class.
     *
     * @access public
     * @throws Exception If arguments are missing when initializing a full widget instance.
     * @param array      $data Widget data. Default is an empty array.
     * @param array|null $args Optional. Widget default arguments. Default is null.
     *
     * @return void
     **/
    public function __construct( $data = [], $args = null ) {

        parent::__construct( $data, $args );

        wp_register_style( 'mdp-ader-elementor-admin', UnityPlugin::get_url() . 'css/elementor-admin' . UnityPlugin::get_suffix() . '.css', [], UnityPlugin::get_version() );
        wp_register_style( 'mdp-ader-elementor', UnityPlugin::get_url() . 'css/ader-elementor' . UnityPlugin::get_suffix() . '.css', [], UnityPlugin::get_version() );

    }

    /**
     * Return a widget name.
     *
     * @return string
     **/
    public function get_name() {

        return 'mdp-ader-custom-add-to-cart-elementor';

    }

    /**
     * Return the widget title that will be displayed as the widget label.
     *
     * @return string
     **/
    public function get_title() {

        return esc_html__( 'Custom Add To Cart', 'ader-elementor' );

    }

    /**
     * Set the widget icon.
     *
     * @return string
     */
    public function get_icon() {

        return 'mdp-custom-add-to-cart-elementor-widget-icon';

    }

    /**
     * Set the category of the widget.
     *
     * @return array with category names
     **/
    public function get_categories() {

        return [ 'general' ];

    }

    /**
     * Get widget keywords. Retrieve the list of keywords the widget belongs to.
     *
     * @access public
     *
     * @return array Widget keywords.
     **/
    public function get_keywords() {

        return [ 'Merkulove', 'Ader', 'cart', 'add', 'woocommerce' ];

    }

    /**
     * Get style dependencies.
     * Retrieve the list of style dependencies the widget requires.
     *
     * @access public
     *
     * @return array Widget styles dependencies.
     **/
    public function get_style_depends() {

        return [ 'mdp-ader-elementor', 'mdp-ader-elementor-admin', 'elementor-icons-fa-solid' ];

    }

	/**
	 * Get script dependencies.
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @access public
     *
	 * @return array Element scripts dependencies.
	 **/
	public function get_script_depends() {

		return [ 'mdp-ader' ];

    }

    /**
     * Add the widget controls.
     *
     * @access protected
     * @return void with category names
     **/
    protected function _register_controls() {

        /** Content Tab. */
        $this->tab_content();

        /** Style Tab. */
        $this->tab_style();

    }

    /**
     * Add widget controls on Content tab.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function tab_content() {

        /** Content -> General Content Section. */
        $this->section_content_general();

        /** Content -> Button Content Section. */
        $this->section_content_button();

        /** Content -> View Cart Link Content Section. */
        $this->section_content_view_cart_link();

    }

    /**
     * Add widget controls on Style tab.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function tab_style() {

        /** Style -> Section Style Button. */
        $this->section_style_button();

        $this->section_style_view_cart_link();

    }

    /**
     * Checks if woocommerce is active
     *
     * @since 1.0.0
     * @access private
     *
     * @return bool
     **/
    private function is_active_woocommerce() {

        $woocommerce_active = false;
        if ( in_array('woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
            $woocommerce_active = true;
        }

        return $woocommerce_active;
    }

    /**
     * Add widget controls: Content -> General Content Section.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_content_general() {

        $this->start_controls_section( 'section_content_general', [
            'label' => esc_html__( 'General', 'ader-elementor' ),
            'tab'   => Controls_Manager::TAB_CONTENT
        ] );

        $products_list = [];

        if ( $this->is_active_woocommerce() ) {
            $products = wc_get_products( ['return' => 'ids', 'limit' => -1] );
            foreach ( $products as $product ) {
                $products_list[$product] = get_the_title( $product );
            }
        }

        $this->add_control(
            'products',
            [
                'label' => esc_html__( 'Products', 'ader-elementor' ),
                'type' => Controls_Manager::SELECT2,
                'multiple' => false,
                'options' => $products_list,
                'default' => !empty( $products_list ) ? array_keys( $products_list )[0] : ''
            ]
        );

        $this->add_control(
            'quantity',
            [
                'label' => esc_html__( 'Quantity', 'ader-elementor' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 100,
                'step' => 1,
                'default' => 1,
            ]
        );

        $this->end_controls_section();

    }


    /**
     * Add widget controls: Content -> Button Content Section.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_content_button()
    {

        $this->start_controls_section( 'section_content_button', [
            'label' => esc_html__('Button', 'ader-elementor'),
            'tab' => Controls_Manager::TAB_CONTENT
        ] );

        $this->add_control(
            'button_text',
            [
                'label' => esc_html__( 'Button content', 'ader-elementor' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'placeholder' => esc_html__( 'Type your inner text here', 'ader-elementor' ),
                'dynamic' => [
                    'active' => true,
                ],
                'default' => esc_html__( 'Add to cart', 'ader-elementor' ),
            ]
        );

        $this->add_control(
            'button_text_alignment',
            [
                'label' => esc_html__( 'Button text alignment', 'ader-elementor' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'ader-elementor' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'ader-elementor' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'ader-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'default' => 'center',
                'toggle' => true,
            ]
        );


        $this->add_control(
            'show_product_price',
            [
                'label' => esc_html__( 'Show product price', 'ader-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'ader-elementor' ),
                'label_off' => esc_html__( 'No', 'ader-elementor' ),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );


        $this->add_control(
            'show_icon',
            [
                'label' => esc_html__( 'Icon', 'ader-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'ader-elementor' ),
                'label_off' => esc_html__( 'No', 'ader-elementor' ),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'button_icon',
            [
                'label' => esc_html__( 'Icon', 'ader-elementor' ),
                'type' => Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-shopping-cart',
                    'library' => 'solid',
                ],
                'condition' => [
                    'show_icon' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'button_icon_position',
            [
                'label' => esc_html__( 'Icon position', 'ader-elementor' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'before',
                'options' => [
                    'before' => esc_html__( 'Before', 'ader-elementor' ),
                    'after' => esc_html__( 'After', 'ader-elementor' ),
                    'above-left' => esc_html__( 'Above left', 'ader-elementor' ),
                    'above-center' => esc_html__( 'Above center', 'ader-elementor' ),
                    'above-right' => esc_html__( 'Above right', 'ader-elementor' ),
                    'under-left' => esc_html__( 'Under left', 'ader-elementor' ),
                    'under-center' => esc_html__( 'Under center', 'ader-elementor' ),
                    'under-right' => esc_html__( 'Under right', 'ader-elementor' ),
                ],
                'condition' => [
                    'show_icon' => 'yes'
                ]
            ]
        );


        $this->add_control(
            'button_alignment',
            [
                'label' => esc_html__( 'Button alignment', 'ader-elementor' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'ader-elementor' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'ader-elementor' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'ader-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                    'justify' => [
                        'title' => esc_html__( 'Justify', 'ader-elementor' ),
                        'icon' => 'fa fa-align-justify',
                    ],
                ],
                'default' => 'center',
                'toggle' => true,
            ]
        );

        $this->add_control(
            'hover_animation',
            [
                'label' => esc_html__( 'Hover Animation', 'ader-elementor' ),
                'type' => Controls_Manager::HOVER_ANIMATION,
            ]
        );

        $this->end_controls_section();

    }

    /**
     * Add widget controls: Content -> View Cart Link Content Section.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_content_view_cart_link() {
        $this->start_controls_section( 'section_content_view_cart_link', [
            'label' => esc_html__( 'View cart link', 'ader-elementor' ),
            'tab' => Controls_Manager::TAB_CONTENT
        ] );

        $this->add_control(
            'view_cart_link_position',
            [
                'label' => esc_html__( 'View cart link position', 'ader-elementor' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'before',
                'options' => [
                    'before' => esc_html__( 'Before', 'ader-elementor' ),
                    'after' => esc_html__( 'After', 'ader-elementor' ),
                    'above-left' => esc_html__( 'Above left', 'ader-elementor' ),
                    'above-center' => esc_html__( 'Above center', 'ader-elementor' ),
                    'above-right' => esc_html__( 'Above right', 'ader-elementor' ),
                    'under-left' => esc_html__( 'Under left', 'ader-elementor' ),
                    'under-center' => esc_html__( 'Under center', 'ader-elementor' ),
                    'under-right' => esc_html__( 'Under right', 'ader-elementor' ),
                ],
            ]
        );

        $this->end_controls_section();

    }

        /**
     * Add widget controls: Style -> Section Style Button.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_button() {

        $this->start_controls_section( 'section_style_button', [
            'label' => esc_html__( 'Button', 'ader-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE
        ] );

        $this->add_responsive_control(
            'section_style_button_margin',
            [
                'label' => esc_html__( 'Margin', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'selectors' => [
                    "{{WRAPPER}} .mdp-ader-elementor-button" => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'section_style_button_padding',
            [
                'label' => esc_html__( 'Padding', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'separator' => 'after',
                'selectors' => [
                    "{{WRAPPER}} .mdp-ader-elementor-button" => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => esc_html__( 'Icon size', 'ader-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'separator' => 'before',
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 20,
                ],
                'selectors' => [
                    "{{WRAPPER}} .mdp-ader-elementor-button-icon" => 'font-size: {{SIZE}}{{UNIT}}; width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'show_icon' => 'yes'
                ]
            ]
        );

        $this->add_responsive_control(
            'icon_spacing',
            [
                'label' => esc_html__( 'Icon spacing', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'separator' => 'after',
                'condition' => [
                    'show_icon' => 'yes'
                ],
                'selectors' => [
                    "{{WRAPPER}} .mdp-ader-elementor-button-icon" => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'price_spacing',
            [
                'label' => esc_html__( 'Price spacing', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'condition' => [
                    'show_product_price' => 'yes'
                ],
                'selectors' => [
                    "{{WRAPPER}} .mdp-ader-elementor-product-price" => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );


        $this->add_control(
            'price_spacing_separator',
            [
                'type' => Controls_Manager::DIVIDER,
                'condition' => [
                        'show_product_price' => 'yes'
                ]
            ]
        );


        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'button_title_typography',
                'label' => esc_html__( 'Typography', 'ader-elementor' ),
                'scheme' =>  Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .mdp-ader-elementor-button-text, {{WRAPPER}} .mdp-ader-elementor-product-price',
            ]
        );

        $this->start_controls_tabs('control_style_tabs');

        $this->start_controls_tab( 'normal_style_tab', ['label' => esc_html__( 'NORMAL', 'ader-elementor' )] );

        $this->add_control(
            'button_text_color',
            [
                'label' => esc_html__( 'Text color', 'ader-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' => Color::get_type(),
                    'value' => Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-ader-elementor-button-text' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label' => esc_html__( 'Icon color', 'ader-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' => Color::get_type(),
                    'value' => Color::COLOR_3,
                ],
                'condition' => [
                    'show_icon' => 'yes'
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-ader-elementor-button-icon' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'price_color',
            [
                'label' => esc_html__( 'Price color', 'ader-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' => Color::get_type(),
                    'value' => Color::COLOR_3,
                ],
                'condition' => [
                    'show_product_price' => 'yes'
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-ader-elementor-product-price' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'button_background',
                'label' => esc_html__( 'Background type', 'ader-elementor' ),
                'types' => [ 'classic', 'gradient', 'video' ],
                'selector' => '{{WRAPPER}} .mdp-ader-elementor-button',
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'border_button_normal',
                'label' => esc_html__( 'Border Type', 'ader-elementor' ),
                'separator' => 'before',
                'selector' => '{{WRAPPER}} .mdp-ader-elementor-button',
            ]
        );

        $this->add_responsive_control(
            'button_border_radius_normal',
            [
                'label' => esc_html__( 'Border radius', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-ader-elementor-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',

                ],
            ]
        );


        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'button_box_shadow_normal',
                'label' => esc_html__( 'Box Shadow', 'ader-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-ader-elementor-button',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab( 'hover_style_tab', ['label' => esc_html__( 'HOVER', 'ader-elementor' )] );

        $this->add_control(
            'button_text_color_hover',
            [
                'label' => esc_html__( 'Text color', 'ader-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' => Color::get_type(),
                    'value' => Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-ader-elementor-button:hover .mdp-ader-elementor-button-text' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'icon_color_hover',
            [
                'label' => esc_html__( 'Icon color', 'ader-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' => Color::get_type(),
                    'value' => Color::COLOR_3,
                ],
                'condition' => [
                    'show_icon' => 'yes'
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-ader-elementor-button:hover .mdp-ader-elementor-button-icon' => 'color: {{VALUE}}',
                ]
            ]
        );

        $this->add_control(
            'price_color_hover',
            [
                'label' => esc_html__( 'Price color', 'ader-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' => Color::get_type(),
                    'value' => Color::COLOR_3,
                ],
                'condition' => [
                    'show_product_price' => 'yes'
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-ader-elementor-button:hover .mdp-ader-elementor-product-price' => 'color: {{VALUE}}',
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'button_background_hover',
                'label' => esc_html__( 'Background type', 'ader-elementor' ),
                'types' => [ 'classic', 'gradient', 'video' ],
                'selector' => '{{WRAPPER}} .mdp-ader-elementor-button:hover',
            ]
        );


        $this->add_control(
            'button_hover_transition',
            [
                'label' => esc_html__( 'Hover transition', 'ader-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 's' ],
                'range' => [
                    's' => [
                        'min' => 0.1,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 's',
                    'size' => 0,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-ader-elementor-button, {{WRAPPER}} .mdp-ader-elementor-button .mdp-ader-elementor-button-icon, {{WRAPPER}} .mdp-ader-elementor-button .mdp-ader-elementor-button-text, {{WRAPPER}} .mdp-ader-elementor-product-price' => 'transition: color {{SIZE}}{{UNIT}}, background {{SIZE}}{{UNIT}}, box-shadow {{SIZE}}{{UNIT}}, border-radius {{SIZE}}{{UNIT}}, border {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'border_button_hover',
                'label' => esc_html__( 'Border Type', 'ader-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-ader-elementor-button:hover',
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'button_border_radius_hover',
            [
                'label' => esc_html__( 'Border radius', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-ader-elementor-button:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',

                ],
            ]
        );


        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'button_box_shadow_hover',
                'label' => esc_html__( 'Box Shadow', 'ader-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-ader-elementor-button:hover',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

    }


    /**
     * Add widget controls: Style -> Section Style View Cart Link.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_view_cart_link()
    {

        $this->start_controls_section( 'section_style_view_cart_link', [
            'label' => esc_html__( 'View cart link', 'ader-elementor' ),
            'tab' => Controls_Manager::TAB_STYLE
        ] );


        $this->add_responsive_control(
            'section_style_cart_link_margin',
            [
                'label' => esc_html__( 'Margin', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'selectors' => [
                    "{{WRAPPER}} .mdp-ader-elementor-box .added_to_cart.wc-forward" => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'section_style_cart_link_padding',
            [
                'label' => esc_html__( 'Padding', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'selectors' => [
                    "{{WRAPPER}} .mdp-ader-elementor-box .added_to_cart.wc-forward" => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );


        $this->add_control(
            'section_style_cart_link_margin_padding_divider',
            [
                'type' => Controls_Manager::DIVIDER,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'view_cart_link_typography',
                'label' => esc_html__( 'Typography', 'ader-elementor' ),
                'scheme' =>  Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .added_to_cart',
            ]
        );

        $this->start_controls_tabs( 'control_link_style_tabs' );

        $this->start_controls_tab( 'normal_link_style_tab', ['label' => esc_html__( 'NORMAL', 'ader-elementor' )] );

        $this->add_control(
            'link_text_color',
            [
                'label' => esc_html__( 'Text color', 'ader-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' => Color::get_type(),
                    'value' => Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .added_to_cart' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'view_cart_link_background',
                'label' => esc_html__( 'Background type', 'ader-elementor' ),
                'types' => [ 'classic', 'gradient', 'video' ],
                'selector' => '{{WRAPPER}} .added_to_cart',
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'border_view_cart_link_normal',
                'label' => esc_html__( 'Border Type', 'ader-elementor' ),
                'separator' => 'before',
                'selector' => '{{WRAPPER}} .added_to_cart',
            ]
        );

        $this->add_responsive_control(
            'view_cart_link_border_radius_normal',
            [
                'label' => esc_html__( 'Border radius', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .added_to_cart' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',

                ],
            ]
        );


        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'view_cart_link_box_shadow_normal',
                'label' => esc_html__( 'Box Shadow', 'ader-elementor' ),
                'selector' => '{{WRAPPER}} .added_to_cart',
            ]
        );


        $this->end_controls_tab();

        $this->start_controls_tab( 'hover_link_style_tab', ['label' => esc_html__( 'HOVER', 'ader-elementor' )] );

        $this->add_control(
            'link_text_color_hover',
            [
                'label' => esc_html__( 'Text color', 'ader-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' => Color::get_type(),
                    'value' => Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .added_to_cart:hover' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'view_cart_link_background_hover',
                'label' => esc_html__( 'Background type', 'ader-elementor' ),
                'types' => [ 'classic', 'gradient', 'video' ],
                'selector' => '{{WRAPPER}} .added_to_cart:hover',
            ]
        );

        $this->add_control(
            'link_hover_transition',
            [
                'label' => esc_html__( 'Hover transition', 'ader-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 's' ],
                'range' => [
                    's' => [
                        'min' => 0.1,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 's',
                    'size' => 0,
                ],
                'selectors' => [
                    '{{WRAPPER}} .added_to_cart' => 'transition: color {{SIZE}}{{UNIT}}, background {{SIZE}}{{UNIT}}, box-shadow {{SIZE}}{{UNIT}}, border-radius {{SIZE}}{{UNIT}}, border {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'border_view_cart_link_hover',
                'label' => esc_html__( 'Border Type', 'ader-elementor' ),
                'selector' => '{{WRAPPER}} .added_to_cart:hover',
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'view_cart_link_border_radius_hover',
            [
                'label' => esc_html__( 'Border radius', 'ader-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .added_to_cart:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',

                ],
            ]
        );


        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'view_cart_link_box_shadow_hover',
                'label' => esc_html__( 'Box Shadow', 'ader-elementor' ),
                'selector' => '{{WRAPPER}} .added_to_cart:hover',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

    }

    /**
     * Render Frontend Output. Generate the final HTML on the frontend.
     *
     * @access protected
     *
     * @return void
     **/
    protected function render() {
        $settings = $this->get_settings_for_display();

        if ( $this->is_active_woocommerce() ) {

            if ( !empty( $settings['products'] ) ) {
                $product_id = esc_sql( $settings['products'] );
            } elseif ( wp_doing_ajax() ) {
                $product_id = esc_sql( $_POST['post_id'] );
            } else {
                $product_id = get_queried_object_id();
            }

            $product = wc_get_product( $product_id );

        }

        ?>
        <!-- Start Ader for Elementor WordPress Plugin -->
        <?php if ( $this->is_active_woocommerce() ): ?>
        <div class="mdp-ader-elementor-box mdp-ader-elementor-button-align-<?php esc_attr_e( $settings['button_alignment'] ); ?>">
            <?php if ( $product ): ?>
            <div class="mdp-ader-elementor-button-wrapper mdp-ader-elementor-view-cart-link-position-<?php esc_attr_e( $settings['view_cart_link_position'] ); ?>">
            <a href="<?php echo esc_url( $product->add_to_cart_url() )?>"
               data-product_id="<?php esc_attr_e( $product->get_id() ) ?>"
               data-quantity="<?php esc_attr_e( isset( $settings['quantity'] ) ? $settings['quantity'] : 1 ); ?>"
               rel="nofollow"
               class="mdp-ader-elementor-button <?php if ( $product->supports( 'ajax_add_to_cart' ) ): ?>ajax_add_to_cart <?php endif; ?><?php if ( $product->is_purchasable() && $product->is_in_stock() ): ?>add_to_cart_button<?php endif; ?> mdp-ader-elementor-button-text-align-<?php esc_attr_e( $settings['button_text_alignment'] ); ?> elementor-animation-<?php esc_attr_e( $settings['hover_animation'] ); ?>">
                 <div class="mdp-ader-elementor-button-content-wrapper <?php if ( $settings['show_icon'] === 'yes' ): ?>mdp-ader-elementor-icon-position-<?php esc_attr_e( $settings['button_icon_position'] ); ?><?php endif; ?>">
                    <?php if ( $settings['show_icon'] === 'yes' ): ?>
                        <div class="mdp-ader-elementor-button-icon">
                            <?php Icons_Manager::render_icon( $settings['button_icon'], [ 'aria-hidden' => 'true' ] ); ?>
                        </div>
                    <?php endif; ?>
                    <div class="mdp-ader-elementor-button-text">
                        <?php esc_html_e( $settings['button_text'] ); ?>
                        <?php if ( $settings['show_product_price'] === 'yes' ): ?>
                            <div class="mdp-ader-elementor-product-price">
                                <?php esc_html_e( $product->get_price() ).esc_html_e( get_woocommerce_currency_symbol() ); ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </a>
            </div>
            <?php else: ?>
            Please set a valid Product!
            <?php endif; ?>
        </div>
        <?php else: ?>
        Install WooCommerce to display the button!
        <?php endif; ?>
        <!-- End Ader for Elementor WordPress Plugin -->
	    <?php

    }

    /**
     * Return link for documentation
     * Used to add stuff after widget
     *
     * @access public
     *
     * @return string
     **/
    public function get_custom_help_url() {

        return 'https://docs.merkulov.design/tag/ader';

    }

}
